﻿/* global window alert jQuery */
/** @class */
var Grid = {
    xhr: null,

    /**
     * Reload the data in the grid from a data source.
     * @method
     * @param {object} params - An object that contains a list with parameters that are going to be send to the server.
     * @fires Grid#beforeEmptyRowInsert, Grid#dataBinding, Grid#dataBound, Grid#cellDataBound
     * @return void
     * @example <input type="text" id="txtSearch">
     * <button id="btnSearch">Search</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnSearch").on("click", function () {
     *         grid.reload({ searchString: $("#txtSearch").val() });
     *     });
     * </script>
     */
    reload: function (params) {
        var data, ajaxOptions, records;
        data = this.data('grid');
        $.extend(data.params, params);
        Grid.StartLoading(this);
        Grid.ReloadHeader(this, data);
        if (data.loader.url) {
            if (!data.loader.data) {
                data.loader.data = {};
            }
            $.extend(data.loader.data, data.params);
            ajaxOptions = $.extend(true, {}, data.loader); //clone loader object
            if (ajaxOptions.dataType === "json" && typeof (ajaxOptions.data) === "object") {
                ajaxOptions.data = JSON.stringify(ajaxOptions.data);
            }
            if (this.xhr) {
                this.xhr.abort();
            }
            this.xhr = $.ajax(ajaxOptions);
        } else if (data.loader.data) {
            records = Grid.GetRecords(data, data.loader.data);
            Grid.LoadData(this, data, records);
        }
        return this;
    },

    /**
     * Clear the content in the grid.
     * @method
     * @return void
     * @example <button id="btnClear">Clear</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnClear").on("click", function () {
     *         grid.clear();
     *     });
     * </script>
     */
    clear: function () {
        var emptyResponse = {}, data = this.data('grid');
        if ("checkbox" === data.selectionMethod) {
            this.find("input#checkAllBoxes").hide();
        }
        emptyResponse[data.mapping.totalRecordsField] = 0;
        this.children("tbody").empty();
        Grid.StopLoading(this);
        Grid.AppendEmptyRow(this, "&nbsp;");
        Grid.ReloadPager(this, data, emptyResponse);
        return this;
    },

    /**
     * Return the number of records presented on the screen.
     * @method
     * @return int
     * @example <button id="btnShowCount">Show Count</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnShowCount").on("click", function () {
     *         alert(grid.count());
     *     });
     * </script>
     */
    count: function () {
        return $(this).find("tbody tr").length;
    },

    /**
     * Render data in the grid
     * @method
     * @param {object} response - An object that contains the data that needs to be loaded in the grid.
     * @fires Grid#beforeEmptyRowInsert, Grid#dataBinding, Grid#dataBound, Grid#cellDataBound
     * @return void
     * @example <table id="grid"></table>
     * <script>
     *     var grid, onSuccessFunc; 
     *     onSuccessFunc = function (response) { 
     *         grid.render(response);
     *     };
     *     grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers", success: onSuccessFunc },
     *         columns: [ { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     * </script>
     */
    render: function (response) {
        var data, records;
        if (!response) {
            return;
        }
        data = this.data('grid');
        if (data) {
            records = Grid.GetRecords(data, response);
            Grid.LoadData(this, data, records);
            Grid.ReloadPager(this, data, response);
        }
    },
    
    /**
     * Remove the grid from the HTML dom tree.
     * @method
     * @return void
     * @example <button id="btnRemove">Remove</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnRemove").on("click", function () {
     *         grid.remove();
     *     });
     * </script>
     */
    remove: function () {
        $(window).unbind('.grid');
        if (this.parent().hasClass("ui-grid-wrapper")) {
            this.unwrap();
        }
        this.removeData();
        this.remove();
    },


    /**
     * Destroy the grid from the HTML dom tree and leave just the "table" tag in the HTML dom tree.
     * @method
     * @return void
     * @example <button id="btnDestroy">Destroy</button>
     * <button id="btnCreate">Create</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid, createFunc;
     *     createFunc = function() {
     *         grid = $("#grid").grid({
     *             loader: { url: "/version_0_3/Demos/GetPlayers" },
     *             columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *         });
     *     };
     *     createFunc();
     *     $("#btnDestroy").on("click", function () {
     *         grid.destroy();
     *     });
     *     $("#btnCreate").on("click", function () {
     *         createFunc();
     *     });
     * </script>
     */
    destroy: function () {
        $(window).unbind('.grid');
        this.removeClass().empty();
        if (this.parent().hasClass("ui-grid-wrapper")) {
            this.unwrap();
        }
        this.removeData();
        this.off();
    },
    
    /**
     * Select a row from the grid based on id parameter.
     * @method
     * @param {string} id - The id of the row that needs to be selected
     * @return void
     * @example <input type="text" id="txtNumber" value="1" />
     * <button id="btnSelect">Select</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         selectionMethod: "checkbox"
     *     });
     *     $("#btnSelect").on("click", function () {
     *         grid.setSelected($("#txtNumber").val());
     *     });
     * </script>
     */
    setSelected: function (id) {
        var $row = Grid.GetRowById(this, id);
        if ($row) {
            Grid._SetSelected(this, $row, id);
        }
    },

    /**
     * Return the id of the selected record.
     * If the multiple selection method is one this method is going to return only the id of the first selected record.
     * @method
     * @return string
     * @example <button id="btnShowSelection">Show Selection</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         selectionMethod: "checkbox"
     *     });
     *     $("#btnShowSelection").on("click", function () {
     *         alert(grid.getSelected());
     *     });
     * </script>
     */
    getSelected: function () {
        return Grid._GetSelected(this);
    },

    /**
     * Return an array with the ids of the selected record.
     * @method
     * @return array
     * @example <button id="btnShowSelection">Show Selections</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         selectionMethod: "checkbox",
     *         selectionType: "multiple"
     *     });
     *     $("#btnShowSelection").on("click", function () {
     *         var selections = grid.getSelections();
     *         $.each(selections, function() {
     *             alert(this);
     *         });
     *     });
     * </script>
     */
    getSelections: function () {
        return Grid._GetSelections(this);
    },
    
    /**
     * Select all records from the grid.
     * @method
     * @return void
     * @example <button id="btnSelectAll">Select All</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         selectionMethod: "checkbox",
     *         selectionType: "multiple"
     *     });
     *     $("#btnSelectAll").on("click", function () {
     *         grid.selectAll();
     *     });
     * </script>
     */
    selectAll: function () {
        var $grid = this,
            data = this.data('grid');
        $grid.find("thead input#checkAllBoxes").prop("checked", true);
        $grid.find("tbody tr").each(function () {
            Grid.SelectRow($grid, data, $(this));
        });
    },
    
    /**
     * Unselect all records from the grid.
     * @method
     * @return void
     * @example <button id="btnSelectAll">Select All</button>
     * <button id="btnUnSelectAll">UnSelect All</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         selectionMethod: "checkbox",
     *         selectionType: "multiple"
     *     });
     *     $("#btnSelectAll").on("click", function () {
     *         grid.selectAll();
     *     });
     *     $("#btnUnSelectAll").on("click", function () {
     *         grid.unSelectAll();
     *     });
     * </script>
     */
    unSelectAll: function () {
        var $grid = $(this),
            data = this.data('grid');
        this.find("thead input#checkAllBoxes").prop("checked", false);
        this.find("tbody tr").each(function () {
            Grid.UnselectRow($grid, data, $(this));
        });
    },
    
    /**
     * Return record by id of the record.
     * @method
     * @param {string} id - The id of the row that needs to be returned.
     * @return object
     * @example <button id="btnGetData">Get Data</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
     *         dataKey: "ID" //define the name of the column that you want to use as ID here.
     *     });
     *     $("#btnGetData").on("click", function () {
     *         var data = grid.getRecordById("2");
     *         alert(data.Name + " born in " + data.PlaceOfBirth);
     *     });
     * </script>
     */
    getRecordById: function (id) {
        return Grid.GetRecordById(this, id);
    },

    /**
     * Return record by index/position of the record.
     * @method
     * @param {int} index - The index/position of the row that needs to be return.
     * @return object
     * @example <button id="btnGetData">Get Data</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnGetData").on("click", function () {
     *         var data = grid.getRecordByIndex(3);
     *         alert(data.Name + " born in " + data.PlaceOfBirth);
     *     });
     * </script>
     */
    getRecordByIndex: function (index) {
        return Grid.GetRecordByIndex(this, index);
    },

    /**
     * Return an array with all records presented in the grid.
     * @method
     * @return array
     * @example <button id="btnGetAllName">Get All Names</button>
     * <br/><br/>
     * <table id="grid"></table>
     * <script>
     *     var grid = $("#grid").grid({
     *         loader: { url: "/version_0_3/Demos/GetPlayers" },
     *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
     *     });
     *     $("#btnGetAllName").on("click", function () {
     *         var records = grid.getAll(), names = "";
     *         $.each(records, function () { 
     *             names += this.record.Name + "(id=" + this.id + "),";
     *         });
     *         alert(names);
     *     });
     * </script>
     */
    getAll: function () {
        return Grid.GetAll(this);
    },

    setCellContent: function (id, index, value) {
        Grid.SetCellContent(this, id, index, value);
    },

    showColumn: function (index) {
        var data, position;
        data = this.data('grid');
        position = Grid.GetColumnPosition(this, index);
        data.columns[position].hidden = false;
        return this;
    },

    hideColumn: function (index) {
        var data, position;
        data = this.data('grid');
        position = Grid.GetColumnPosition(this, index);
        data.columns[position].hidden = true;
        return this;
    },

    

    init: function (customOptions) {
        var options;
        if (!this.data('grid')) {
            options = Grid.SetOptions(this, customOptions);
            Grid.InitGrid(this);
            Grid.CreateHeader(this);
            Grid.AppendEmptyRow(this, "&nbsp;");
            Grid.CreatePager(this);
            if (options.autoLoad) {
                this.reload();
            }
        }
        return this;
    },
    SetOptions: function ($grid, customOptions) {
        var options;
        options = Grid.GetDefaultOptions($grid);
        if (customOptions.uiLibrary && customOptions.uiLibrary === "bootstrap") {
            $.extend(true, options, Grid.GetBootstrapOptions());
        }
        $.extend(true, options, customOptions);
        $grid.data('grid', options);
        return options;
    },

    GetDefaultOptions: function ($grid) {
        return {
            loader: {
                /** The type of the ajax request to the server.
                  * @alias loader.type
                  * @memberof Grid
                  * @type (GET|POST)
                  * @default "GET"
                  * @example <table id="grid"></table>
                  * <script>
                  *     $("#grid").grid({
                  *         loader: { url: "/version_0_3/Demos/GetPlayers", type: "POST" },
                  *         columns: [ { index: "ID" }, { index: "Name" } ]
                  *     });
                  * </script>
                  */
                type: "GET",
                /** The url to the end-point that will return a json object with data for the grid.
                  * @alias loader.url
                  * @memberof Grid
                  * @type string
                  * @default undefined
                  * @example <table id="grid"></table>
                  * <script>
                  *     $("#grid").grid({
                  *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                  *         columns: [ { index: "ID" }, { index: "Name" } ]
                  *     });
                  * </script>
                  */
                url: undefined,
                /** An array that is going to be in use by the grid as a data source, if the url setting is not configured.
                  * @alias loader.data
                  * @memberof Grid
                  * @type array
                  * @default undefined
                  * @example <table id="grid"></table>
                  * <script>
                  *     $("#grid").grid({
                  *         loader: { url: "/version_0_3/Demos/GetPlayers", data: { searchString: "Bulgaria" } },
                  *         columns: [ { index: "Name" }, { index: "PlaceOfBirth" } ]
                  *     });
                  * </script>
                  */
                data: undefined,
                /** A function that is executed after successful loading of the data from the server.
                  * @alias loader.success
                  * @memberof Grid
                  * @type function
                  * @default Grid.LoaderSuccessHandler($grid)
                  * @example <table id="grid"></table>
                  * <script>
                  *     var grid, onSuccessFunc; 
                  *     onSuccessFunc = function (response) { 
                  *         alert("The result contains " + response.records.length + " records.");
                  *         grid.render(response);
                  *     };
                  *     grid = $("#grid").grid({
                  *         loader: { url: "/version_0_3/Demos/GetPlayers", success: onSuccessFunc },
                  *         columns: [ { index: "Name" }, { index: "PlaceOfBirth" } ]
                  *     });
                  * </script>
                  */
                success: Grid.LoaderSuccessHandler($grid)
            },
            /** An array that holds the configurations of each column from the grid.
              * @memberof Grid
              * @type array
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         columns: [ { index: "ID", width: 30 }, { index: "Name" }, { index: "PlaceOfBirth", name: "Birth Place" } ]
              *     });
              * </script>
              */
            columns: [
                {
                    /** If set to true the column will not be displayed in the grid. By default all columns are displayed.
                      * @alias columns.hidden
                      * @memberof Grid
                      * @type boolean
                      * @default false
                      */
                    hidden: false,
                    /** The width of the column. Numeric values are treated as pixels.
                      * If the width is undefined the width of the column is not set and depends on the with of the table(grid).
                      * @alias columns.width
                      * @memberof Grid
                      * @type int|string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID", width: 20 },
                      *             { index: "Name", width: 120 },
                      *             { index: "PlaceOfBirth" }
                      *         ]
                      *     });
                      * </script>
                      */                    
                    width: undefined,
                    /** Indicates if the column is sortable.
                      * If set to true the user can click the column header and sort the grid by the column source field.
                      * @alias columns.sortable
                      * @memberof Grid
                      * @type boolean
                      * @default false
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name", sortable: true },
                      *             { index: "PlaceOfBirth", sortable: false },
                      *             { index: "DateOfBirth", type: "date", name: "Birth Date" }
                      *         ]
                      *     });
                      * </script>
                      */
                    sortable: false,
                    /** Indicates the type of the column.
                      * @alias columns.type
                      * @memberof Grid
                      * @type undefined|checkbox|ui-icon|date
                      * @default undefined
                      */
                    type: undefined, //checkbox
                    /** The caption that is going to be displayed in the header of the grid.
                      * @alias columns.name
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name", name: "Player" },
                      *             { index: "PlaceOfBirth", name: "Place of Birth" },
                      *             { index: "DateOfBirth", type: "date", name: "Birth Date" }
                      *         ]
                      *     });
                      * </script>
                      */
                    name: undefined, //TODO: rename to title
                    /** The data source column from the server response.
                      * If the columns.name is not defined this value is used as columns.name.
                      * @alias columns.index
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name" },
                      *             { index: "PlaceOfBirth", name: "Place of Birth" },
                      *             { index: "DateOfBirth", type: "date" }
                      *         ]
                      *     });
                      * </script>
                      */
                    index: undefined, //TODO: rename to dataSource
                    /** This setting control the alignment of the text in the cell.
                      * @alias columns.align
                      * @memberof Grid
                      * @type left|right|center|justify|initial|inherit
                      * @default "left"
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID", align: "center" },
                      *             { index: "Name", align: "right" },
                      *             { index: "PlaceOfBirth", align: "left" }
                      *         ]
                      *     });
                      * </script>
                      */
                    align: "left",
                    /** This setting control the wrapping of the text in the cell.
                      * @alias columns.nowrap
                      * @memberof Grid
                      * @type nowrap|normal
                      * @default "normal"
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID", width: 20 },
                      *             { index: "Name", width: 100, nowrap: "nowrap" },
                      *             { index: "PlaceOfBirth" }
                      *         ]
                      *     });
                      * </script>
                      */
                    nowrap: "normal", //TODO: replace with style or cssClass
                    /** The text for the cell tooltip.
                      * @alias columns.tooltip
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID", tooltip: "This is my tooltip 1." },
                      *             { index: "Name", tooltip: "This is my tooltip 2." },
                      *             { index: "PlaceOfBirth", tooltip: "This is my tooltip 3." }
                      *         ]
                      *     });
                      * </script>
                      */
                    tooltip: undefined,
                    /** Css class that is going to be in use for the cell.
                      * @alias columns.icon
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name" },
                      *             { index: "PlaceOfBirth" },
                      *             { name: "", index: "Edit", width: 20, type: "ui-icon", icon: "ui-icon-pencil", onClick: function (e, data) { alert("name=" + data.record.Name); } }
                      *         ]
                      *     });
                      * </script>
                      */
                    icon: undefined, //TODO: rename the icon to cssClass + add style
                    /** The function that is going to be executed on click.
                      * @alias columns.onClick
                      * @memberof Grid
                      * @type function
                      * @default undefined
                      * @param {object} e - event data
                      * @param {object} data - Data object with id, index and record object for the cell.
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name" },
                      *             { index: "PlaceOfBirth" },
                      *             { name: "", index: "Info", width: 20, type: "ui-icon", icon: "ui-icon-info", onClick: function (e, data) { alert("record with id=" + data.id + " is clicked."); } }
                      *         ]
                      *     });
                      * </script>
                      */
                    onClick: undefined,
                    /** Format the date when the type of the column is date. 
                      * This configuration setting is going to work only if you have implementation of format method for the Date object.
                      * You can use external libraries like http://blog.stevenlevithan.com/archives/date-time-format for that.
                      * @alias columns.format
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script src="http://stevenlevithan.com/assets/misc/date.format.js"></script>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name" },
                      *             { index: "DateOfBirth", type: 'date', format: 'HH:MM:ss mm/dd/yyyy' }
                      *         ]
                      *     });
                      * </script>
                      */
                    format: undefined,
                    /** Number of decimal digits after the decimal point.
                      * @alias columns.decimalDigits
                      * @memberof Grid
                      * @type int
                      * @default undefined
                      */
                    decimalDigits: undefined,
                    /** Template for the content in the column.
                      * Use curly brackets "{}" to wrap the names of data source columns from server response.
                      * @alias columns.tmpl
                      * @memberof Grid
                      * @type string
                      * @default undefined
                      * @example <table id="grid"></table>
                      * <script>
                      *     $("#grid").grid({
                      *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                      *         columns: [
                      *             { index: "ID" },
                      *             { index: "Name" },
                      *             { name: "Info", tmpl: "{Name} is born in {PlaceOfBirth}." }
                      *         ]
                      *     });
                      * </script>
                      */
                    tmpl: undefined
                }
            ],
            mapping: {
                /** The name of the object in the server response, that contains array with records, that needs to be display in the grid.
                  * @alias mapping.dataField
                  * @memberof Grid
                  * @type string
                  * @default "records"
                  */
                dataField: "records",
                /** The name of the object in the server response, that contains the number of all records on the server.
                  * @alias mapping.totalRecordsField
                  * @memberof Grid
                  * @type string
                  * @default "total"
                  */
                totalRecordsField: "total"
            },
            params: {},
            defaultParams: {
                /** The name of the parameter that is going to send the name of the column for sorting.
                  * The "sortable" setting for at least one column should be enabled in order this parameter to be in use.
                  * @alias defaultParams.sortBy
                  * @memberof Grid
                  * @type string
                  * @default "sortBy"
                  */
                sortBy: "sortBy",
                /** The name of the parameter that is going to send the direction for sorting.
                  * The "sortable" setting for at least one column should be enabled in order this parameter to be in use.
                  * @alias defaultParams.direction
                  * @memberof Grid
                  * @type string
                  * @default "direction"
                  */
                direction: "direction",
                /** The name of the parameter that is going to send the number of the page.
                  * The pager should be enabled in order this parameter to be in use.
                  * @alias defaultParams.page
                  * @memberof Grid
                  * @type string
                  * @default "page"
                  */
                page: "page",
                /** The name of the parameter that is going to send the maximum number of records per page.
                  * The pager should be enabled in order this parameter to be in use.
                  * @alias defaultParams.limit
                  * @memberof Grid
                  * @type string
                  * @default "limit"
                  */
                limit: "limit"
            },
            /** The name of the UI library that is going to be in use. 
              * Currently we support only jQuery UI and bootstrap. jQuery UI or Bootstrap should be manually included to the page where the grid is in use.
              * @memberof Grid
              * @type (jqueryui|bootstrap)
              * @default "jqueryui"
              * @example <table id="grid"></table>
              * <link href="//maxcdn.bootstrapcdn.com/bootstrap/3.3.2/css/bootstrap.min.css" rel="stylesheet">
              * <script src="//maxcdn.bootstrapcdn.com/bootstrap/3.3.2/js/bootstrap.min.js"></script>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         uiLibrary: "bootstrap",
              *         columns: [
              *             { index: "ID" },
              *             { index: "Name" },
              *             { index: "PlaceOfBirth" }
              *         ],
              *         pager: { enable: true, limit: 2, sizes: [2, 5, 10, 20] }
              *     });
              * </script>
              */
            uiLibrary: "jqueryui",
            style: {
                wrapper: "gj-grid-wrapper",
                table: "gj-grid-table ui-widget-content gj-grid-ui-table",
                loadingCover: "gj-grid-loading-cover",
                loadingText: "gj-grid-loading-text",
                header: {
                    cell: "ui-widget-header ui-state-default gj-grid-ui-thead-th",
                    sortable: "gj-grid-thead-sortable",
                    sortAscIcon: "gj-grid-ui-thead-th-sort-icon ui-icon ui-icon-arrowthick-1-s",
                    sortDescIcon: "gj-grid-ui-thead-th-sort-icon ui-icon ui-icon-arrowthick-1-n"
                },
                content: {
                    rowHover: "ui-state-hover",
                    rowSelected: "ui-state-active"
                },
                pager: {
                    cell: "ui-widget-header ui-state-default ui-grid-pager-cell",
                    stateDisabled: "ui-state-disabled"
                }
            },

            /** The type of the row selection.<br/>
              * If the type is set to multiple the user will be able to select more then one row from the grid.
              * @memberof Grid
              * @type (single|multiple)
              * @default "single"
              * @example $("table").grid({  });
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         selectionType: "multiple",
              *         selectionMethod: "checkbox",
              *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
              *     });
              * </script>
              */
            selectionType: 'single',

            /** The type of the row selection method.<br/>
              * If this setting is set to "basic" when the user select a row, then this row will be highlighted.<br/>
              * If this setting is set to "checkbox" a column with checkboxes will appear as first row of the grid and when the user select a row, then this row will be highlighted and the checkbox selected.
              * @memberof Grid
              * @type (basic|checkbox)
              * @default "basic"
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         selectionType: "single",
              *         selectionMethod: "checkbox",
              *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
              *     });
              * </script>
              */
            selectionMethod: 'basic',

            /** When this setting is enabled the content of the grid will be loaded automatically after the creation of the grid.
              * @memberof Grid
              * @type boolean
              * @default true
              * @example <table id="grid"></table>
              * <script>
              *     var grid = $("#grid").grid({ 
              *         loader: { url: "/version_0_3/Demos/GetPlayers" }, 
              *         autoLoad: false,
              *         columns: [ { index: "ID" }, { index: "Name" } ]
              *     });
              *     grid.reload(); //call .reload() explicitly in order to load the data in the grid
              * </script>
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({ 
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         autoLoad: true,
              *         columns: [ { index: "ID" }, { index: "Name" } ]
              *     });
              * </script>
              */
            autoLoad: true,

            /** The text that is going to be displayed if the grid is empty.
              * @memberof Grid
              * @type string
              * @default "No records found."
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers", data: { searchString: "sadasd" } },
              *         notFoundText: "No records found custom message",
              *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
              *     });
              * </script>
              */
            notFoundText: "No records found.",

            /** Width of the grid.
              * @memberof Grid
              * @type int
              * @default undefined
              * @example <table id="grid"></table>
              * <script>
              *     $("#grid").grid({
              *         loader: { url: "/version_0_3/Demos/GetPlayers" },
              *         width: 400,
              *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
              *     });
              * </script>
              */
            width: undefined,

            /** Minimum width of the grid.
              * @memberof Grid
              * @type int
              * @default undefined
              */
            minWidth: undefined,

            pager: {
                /** This setting control the visualization of the pager. If this setting is enabled the pager would show.
                  * @alias pager.enable
                  * @memberof Grid
                  * @type boolean
                  * @default false
                  */
                enable: false,

                /** The maximum number of records that can be show by page.
                  * @alias pager.limit
                  * @memberof Grid
                  * @type int
                  * @default 10
                  */
                limit: 10,

                /** Array that contains the possible page sizes of the grid.
                  * When this setting is set, then a drop down with the options for each page size is visualized in the pager.
                  * @alias pager.sizes
                  * @memberof Grid
                  * @type array
                  * @default undefined
                  */
                sizes: undefined,

                /** Array that contains a list with jquery objects that are going to be used on the left side of the pager.
                  * @alias pager.leftControls
                  * @memberof Grid
                  * @type array
                  * @default array
                  */
                leftControls: [
                    $('<div title="First" data-role="page-first" class="ui-icon ui-icon-seek-first ui-grid-icon"></div>'),
                    $('<div title="Previous" data-role="page-previous" class="ui-icon ui-icon-seek-prev ui-grid-icon"></div>'),
                    $('<div>Page</div>'),
                    $('<div></div>').append($('<input type="text" data-role="page-number" class="ui-grid-pager" value="0">')),
                    $('<div>of&nbsp;</div>'),
                    $('<div data-role="page-label-last">0</div>'),
                    $('<div title="Next" data-role="page-next" class="ui-icon ui-icon-seek-next ui-grid-icon"></div>'),
                    $('<div title="Last" data-role="page-last" class="ui-icon ui-icon-seek-end ui-grid-icon"></div>'),
                    $('<div title="Reload" data-role="page-refresh" class="ui-icon ui-icon-refresh ui-grid-icon"></div>'),
                    $('<div></div>').append($('<select data-role="page-size" class="ui-grid-page-sizer"></select>'))
                ],

                /** Array that contains a list with jquery objects that are going to be used on the right side of the pager.
                  * @alias pager.rightControls
                  * @memberof Grid
                  * @type array
                  * @default array
                  */
                rightControls: [
                    $('<div>Displaying records&nbsp;</div>'),
                    $('<div data-role="record-first">0</div>'),
                    $('<div>&nbsp;-&nbsp;</div>'),
                    $('<div data-role="record-last">0</div>'),
                    $('<div>&nbsp;of&nbsp;</div>'),
                    $('<div data-role="record-total">0</div>').css({ "margin-right": "5px" })
                ]
            }
        };
    },

    GetBootstrapOptions: function () {
        return {
            style: {
                wrapper: "gj-grid-wrapper",
                table: "gj-grid-table table table-bordered table-hover",
                header: {
                    cell: "gj-grid-bootstrap-thead-cell",
                    sortable: "gj-grid-thead-sortable",
                    sortAscIcon: "glyphicon glyphicon-sort-by-alphabet",
                    sortDescIcon: "glyphicon glyphicon-sort-by-alphabet-alt"
                },
                content: {
                    rowHover: "",
                    rowSelected: "active"
                },
                pager: {
                    cell: "gj-grid-bootstrap-tfoot-cell",
                    stateDisabled: "ui-state-disabled"
                }
            },
            pager: {
                leftControls: [
                    $('<button type="button" data-role="page-first" title="First Page" class="btn btn-default btn-sm"><span class="glyphicon glyphicon-step-backward"></span></button>'),
                    $('<div>&nbsp;</div>'),
                    $('<button type="button" data-role="page-previous" title="Previous Page" class="btn btn-default btn-sm"><span class="glyphicon glyphicon-backward"></span></button>'),
                    $('<div>&nbsp;</div>'),
                    $('<div>Page</div>'),
                    $('<div>&nbsp;</div>'),
                    $('<div></div>').append($('<input data-role="page-number" class="form-control input-sm" style="width: 40px; text-align: right;" type="text" value="0">')),
                    $('<div>&nbsp;</div>'),
                    $('<div>of&nbsp;</div>'),
                    $('<div data-role="page-label-last">0</div>'),
                    $('<div>&nbsp;</div>'),
                    $('<button type="button" data-role="page-next" title="Next Page" class="btn btn-default btn-sm"><span class="glyphicon glyphicon-forward"></span></button>'),
                    $('<div>&nbsp;</div>'),
                    $('<button type="button" data-role="page-last" title="Last Page" class="btn btn-default btn-sm"><span class="glyphicon glyphicon-step-forward"></span></button>'),
                    $('<div>&nbsp;</div>'),
                    $('<button type="button" data-role="page-refresh" title="Reload" class="btn btn-default btn-sm"><span class="glyphicon glyphicon-refresh"></span></button>'),
                    $('<div>&nbsp;</div>'),
                    $('<div></div>').append($('<select data-role="page-size" class="form-control input-sm"></select></div>'))
                ],
                rightControls: [
                    $('<div>Displaying records&nbsp;</div>'),
                    $('<div data-role="record-first">0</div>'),
                    $('<div>&nbsp;-&nbsp;</div>'),
                    $('<div data-role="record-last">0</div>'),
                    $('<div>&nbsp;of&nbsp;</div>'),
                    $('<div data-role="record-total">0</div>').css({ "margin-right": "5px" })
                ]
            }
        };
    },

    LoaderSuccessHandler: function ($grid) {
        return function (response) {
            $grid.render(response);
        };
    },

    InitGrid: function ($grid) {
        var data = $grid.data('grid');
        $grid.wrapAll('<div class="' + data.style.wrapper + '" />');
        if (data.width) {
            $grid.parent().css("width", data.width);
        }
        if (data.minWidth) {
            $grid.css("min-width", data.minWidth);
        }
        $grid.addClass(data.style.table);
        if (data.pager.enable) {
            data.params[data.defaultParams.page] = 1;
            data.params[data.defaultParams.limit] = data.pager.limit;
            $grid.data('grid', data);
        }
        $grid.append($("<tbody/>"));
    },

    CreateHeaderRow: function ($grid, data) {
        var $row, $cell, columns, len, style, i, sortBy, direction;

        columns = data.columns;
        len = columns.length;
        style = data.style.header;
        sortBy = data.params[data.defaultParams.sortBy];
        direction = data.params[data.defaultParams.direction];

        $row = $("<tr/>");
        for (i = 0; i < len; i += 1) {
            if (!columns[i].hidden) {
                $cell = $("<th/>");
                if (columns[i].width) {
                    $cell.css("width", columns[i].width);
                }
                $cell.addClass(style.cell);
                $cell.css("text-align", columns[i].align || "left");
                if (columns[i].sortable) {
                    $cell.addClass(style.sortable);
                    $cell.bind("click", Grid.CreateSortHandler($grid, $cell));
                }
                if ("checkbox" === data.selectionMethod && "multiple" === data.selectionType && "checkbox" === columns[i].type) {
                    $cell.append($("<input type='checkbox' id='checkAllBoxes' />").hide().click(function () {
                        var isSelect = this.checked;
                        $grid.find("tbody tr").each(function () {
                            if (isSelect) {
                                Grid.SelectRow($grid, data, $(this));
                            } else {
                                Grid.UnselectRow($grid, data, $(this));
                            }
                        });
                    }));
                } else {
                    $cell.append($("<div style='float: left'/>").text(typeof(columns[i].name) === "undefined" ? columns[i].index : columns[i].name));
                    if (sortBy && direction && columns[i].index === sortBy) {
                        $cell.append($("<span style='float: left; margin-left:5px;'/>").addClass("desc" === direction ? style.sortDescIcon : style.sortAscIcon));
                    }
                }

                $cell.data("cell", columns[i]);
                $row.append($cell);
            }
        }

        return $row;
    },

    CreateHeader: function ($grid) {
        var data, $thead, $row;
        data = $grid.data('grid');
        if ("checkbox" === data.selectionMethod) {
            data.columns = [{ name: '', index: data.dataKey, width: 30, align: 'center', type: 'checkbox' }].concat(data.columns);
        }

        $row = Grid.CreateHeaderRow($grid, data);

        $thead = $("<thead />");
        $thead.append($row);
        $grid.prepend($thead);
    },

    ReloadHeader: function ($grid, data) {
        var $row = Grid.CreateHeaderRow($grid, data);
        $grid.children("thead").empty().append($row);
    },

    CreateSortHandler: function ($grid, $cell) {
        return function () {
            var $icon, data, cellData, params = {};
            if ($grid.count() > 1) {
                data = $grid.data('grid');
                cellData = $cell.data("cell");
                cellData.direction = (cellData.direction === "asc" ? "desc" : "asc");
                params[data.defaultParams.sortBy] = cellData.index;
                params[data.defaultParams.direction] = cellData.direction;
                $grid.reload(params);
            }
        };
    },

    StartLoading: function ($grid) {
        var $tbody, $cover, $loading, width, height, top, data;
        Grid.StopLoading($grid);
        data = $grid.data('grid');
        if (0 === $grid.outerHeight()) {
            return;
        }
        $tbody = $grid.children("tbody");
        width = $tbody.outerWidth(false);
        height = $tbody.outerHeight(false);
        top = $tbody.prev().outerHeight(false) + parseInt($grid.parent().css("padding-top").replace('px', ''), 10);
        $cover = $("<div data-role='loading-cover' />").addClass(data.style.loadingCover).css({
            width: width,
            height: height,
            top: top
        });
        $loading = $("<div data-role='loading-text'>Loading...</div>").addClass(data.style.loadingText);
        $loading.insertAfter($grid);
        $cover.insertAfter($grid);
        $loading.css({
            top: top + (height / 2) - ($loading.outerHeight(false) / 2),
            left: (width / 2) - ($loading.outerWidth(false) / 2)
        });
    },

    StopLoading: function ($grid) {
        $grid.parent().find("div[data-role='loading-cover']").remove();
        $grid.parent().find("div[data-role='loading-text']").remove();
    },

    CreateAddRowHoverHandler: function ($row, cssClass) {
        return function () {
            $row.addClass(cssClass);
        };
    },

    CreateRemoveRowHoverHandler: function ($row, cssClass) {
        return function () {
            $row.removeClass(cssClass);
        };
    },

    AppendEmptyRow: function ($grid, caption) {
        var data, $row, $cell, $wrapper;
        data = $grid.data('grid');
        $row = $("<tr data-role='emptyRow'/>");
        $cell = $("<td/>").css({ "width": "100%", "text-align": "center" });
        $cell.attr("colspan", $grid.find("thead > tr > th").length);
        $wrapper = $("<div />").html(caption);
        $cell.append($wrapper);
        $row.append($cell);

        /**
         * Event fires before addition of an empty row to the grid.
         *
         * @event beforeEmptyRowInsert
         * @memberof Grid
         * @property {object} e - event data
         * @property {object} $row - The empty row as jquery object
         * @example <table id="grid"></table>
         * <script>
         *     var grid = $("#grid").grid({
         *         loader: { 
         *             url: "/version_0_3/Demos/GetPlayers",
         *             data: { searchString: "not existing data" } //search for not existing data in order to fire the event
         *         },
         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
         *     });
         *     grid.on("beforeEmptyRowInsert", function (e, $row) {
         *         alert("beforeEmptyRowInsert is fired.");
         *     });
         * </script>
         */
        $grid.trigger("beforeEmptyRowInsert", [$row]);

        $grid.append($row);
    },

    CreateHandler: function (func, data) {
        return function (e) {
            e.stopPropagation();
            func(e, data);
        };
    },

    LoadData: function ($grid, data, records) {
        var data, style, records, i, j, recLen, rowCount, colLen, id,            
            $tbody, $rows, $row, $checkAllBoxes, $cell, $wrapper;

        /**
         * Event fired before data binding takes place.
         *
         * @event dataBinding
         * @memberof Grid
         * @property {object} e - event data
         * @property {array} records - the list of records received from the server
         * @example <table id="grid"></table>
         * <script>
         *     var grid = $("#grid").grid({
         *         loader: { url: "/version_0_3/Demos/GetPlayers" },
         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
         *     });
         *     grid.on("dataBinding", function (e, records) {
         *         alert("dataBinding is fired. " + records.length + " records will be loaded in the grid.");
         *     });
         * </script>
         */
        $grid.trigger("dataBinding", [records]);

        style = data.style.content;
        recLen = records.length;
        colLen = data.columns.length;
        Grid.StopLoading($grid);
        $tbody = $grid.find("tbody");
        if ("checkbox" === data.selectionMethod && "multiple" === data.selectionType) {
            $checkAllBoxes = $grid.find("input#checkAllBoxes");
            $checkAllBoxes.prop("checked", false);
            if (0 === recLen) {
                $checkAllBoxes.hide();
            } else {
                $checkAllBoxes.show();
            }
        }
        $tbody.find("tr[data-role='emptyRow']").remove(); //Remove empty row
        if (0 === recLen) {
            $tbody.empty();
            Grid.AppendEmptyRow($grid, data.notFoundText);
        }

        $rows = $tbody.children("tr");
        rowCount = $rows.length;

        for (i = 0; i < rowCount; i += 1) {
            if (i < recLen) {
                id = (data.dataKey && records[i][data.dataKey]) ? records[i][data.dataKey] : (i + 1);
                $row = $rows.eq(i);
                $row.data("row", { id: id, record: records[i] });
                $row.unbind("click").bind("click", Grid.CreateRowClickHandler($grid, id, records[i]));

                for (j = 0; j < colLen; j += 1) {
                    if (!data.columns[j].hidden) {
                        $wrapper = $row.find("td:eq(" + j + ") div");
                        Grid.CreateCell($grid, $wrapper, data.columns[j], records[i], id, "update");
                    }
                }

                /**
                 * Event fires after insert of a row in the grid during the loading of the data
                 *
                 * @event rowDataBound
                 * @memberof Grid
                 * @property {object} e - event data
                 * @property {object} $row - the row presented as jquery object
                 * @property {object} id - the id of the record
                 * @property {object} record - the data of the row record
                 * @example <table id="grid"></table>
                 * <script>
                 *     var grid = $("#grid").grid({
                 *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                 *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
                 *     });
                 *     grid.on("rowDataBound", function (e, $row, id, record) {
                 *         alert("rowDataBound is fired for row with id=" + id + ".");
                 *     });
                 * </script>
                 */
                $grid.trigger("rowDataBound", [$row, id, records[i]]);
            } else {
                $tbody.find("tr:gt(" + (i - 1) + ")").remove();
                break;
            }
        }

        for (i = rowCount; i < recLen; i += 1) {
            id = (data.dataKey && records[i][data.dataKey]) ? records[i][data.dataKey] : (i + 1);
            $row = $($tbody[0].insertRow(i));
            $row.bind({
                "mouseenter.grid": Grid.CreateAddRowHoverHandler($row, style.rowHover),
                "mouseleave.grid": Grid.CreateRemoveRowHoverHandler($row, style.rowHover)
            });
            $row.data("row", { id: id, record: records[i] });
            $row.bind("click", Grid.CreateRowClickHandler($grid, id, records[i]));
            for (j = 0; j < colLen; j += 1) {
                if (!data.columns[j].hidden) {
                    $cell = $("<td/>").css("text-align", data.columns[j].align || "left");
                    $wrapper = $("<div />").css("white-space", data.columns[j].nowrap ? "nowrap" : "normal");
                    if (data.columns[j].tooltip) {
                        $wrapper.attr("title", data.columns[j].tooltip);
                    }
                    $cell.append($wrapper);
                    $row.append($cell);
                    Grid.CreateCell($grid, $wrapper, data.columns[j], records[i], id, "create");
                }
            }
            $grid.trigger("rowDataBound", [$row, id, records[i]]);
        }

        /**
         * Event fires after the loading of the data in the grid.
         *
         * @event dataBound
         * @memberof Grid
         * @property {object} e - event data
         * @property {array} records - the list of records received from the server
         * @example <table id="grid"></table>
         * <script>
         *     var grid = $("#grid").grid({
         *         loader: { url: "/version_0_3/Demos/GetPlayers" },
         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ]
         *     });
         *     grid.on("dataBound", function (e, records) {
         *         alert("dataBound is fired. " + records.length + " records are bound to the grid.");
         *     });
         * </script>
         */
        $grid.trigger("dataBound", [records]);
    },

    CreateCell: function ($grid, $wrapper, column, record, id, mode) {
        var text;

        if ("checkbox" === column.type) {
            if ("create" === mode) {
                $wrapper.append($("<input />").attr("type", "checkbox").val(record[column.index]).click(function (e) {
                    e.stopPropagation();
                    $grid.setSelected(this.value);
                }));
            } else {
                $wrapper.find("input[type='checkbox']").val(record[column.index]).prop("checked", false);
            }
        } else if ("ui-icon" === column.type) {
            if ("create" === mode) {
                $wrapper.append($("<span/>").addClass("ui-icon").addClass(column.icon).css({ "cursor": "pointer" }));
            }
        } else if (column.tmpl) {
            text = column.tmpl;
            column.tmpl.replace(/\{(.+?)\}/g, function ($0, $1) {
                text = text.replace($0, record[$1]);
            });
            $wrapper.text(text);
        } else {
            Grid.SetCellText($wrapper, column, record[column.index]);
        }
        //remove all event handlers
        if ("update" === mode) {
            $wrapper.off();
        }
        if (column.onClick) {
            $wrapper.on("click", Grid.CreateHandler(column.onClick, { id: id, index: column.index, record: record }));
        }

        /**
         * Event fires after insert of a cell in the grid during the loading of the data
         *
         * @event cellDataBound
         * @memberof Grid
         * @property {object} e - event data
         * @property {object} $wrapper - the cell wrapper presented as jquery object 
         * @property {string} id - the id of the record
         * @property {int} index - the index number of the column
         * @property {object} record - the data of the row record
         * @example <table id="grid"></table>
         * <script>
         *     var grid = $("#grid").grid({
         *         loader: { url: "/version_0_3/Demos/GetPlayers" },
         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" }, { index: "IsBulgarian", name: "Is Bulgarian" } ]
         *     });
         *     grid.on("cellDataBound", function (e, $wrapper, id, index, record) {
         *         if ("IsBulgarian" === index) {
         *             $wrapper.text(record.PlaceOfBirth.indexOf("Bulgaria") > -1 ? "Bulgarian" : "");
         *         }
         *     });
         * </script>
         */
        $grid.trigger("cellDataBound", [$wrapper, id, column.index, record]);
    },

    SetCellText: function ($wrapper, column, value) {
        var text = Grid.FormatText(value, column.type, column.format);
        if (column.decimalDigits && text) {
            text = parseFloat(text).toFixed(column.decimalDigits);
        }
        if (!column.tooltip) {
            $wrapper.attr("title", text);
        }
        $wrapper.text(text);
    },

    FormatText: function (text, type, format) {
        var dt, day, month;
        if (text && type) {
            switch (type) {
                case "date":
                    if (text.indexOf("/Date(") > -1) {
                        dt = new Date(parseInt(text.substr(6), 10));
                    } else {
                        var parts = text.match(/(\d+)/g);
                        // new Date(year, month, date, hours, minutes, seconds);
                        dt = new Date(parts[0], parts[1] - 1, parts[2], parts[3], parts[4], parts[5]); // months are 0-based
                    }

                    if (dt.format && format) {
                        text = dt.format(format); //using 3rd party plugin "Date Format 1.2.3 by (c) 2007-2009 Steven Levithan <stevenlevithan.com>"
                    } else {
                        day = dt.getDate().toString().length === 2 ? dt.getDate() : "0" + dt.getDate();
                        month = (dt.getMonth() + 1).toString();
                        month = month.length === 2 ? month : "0" + month;
                        text = month + "/" + day + "/" + dt.getFullYear();
                    }
                    break;
                case "money":
                    text = parseFloat(text).toFixed(2);
                    break;
            }
        } else {
            text = (typeof (text) === "undefined" || text === null) ? "" : text.toString();
        }
        return text;
    },

    GetRecords: function (data, response) {
        var records = [];
        if ($.isArray(response)) {
            records = response;
        } else if (data && data.mapping && $.isArray(response[data.mapping.dataField])) {
            records = response[data.mapping.dataField];
        }
        return records;
    },

    CreateRowClickHandler: function ($grid, id, record) {
        return function (e) {
            Grid._SetSelected($grid, $(this), id);
        };
    },

    GetRowById: function ($grid, id) {
        var rowData, i, len, $row,
            rows = $("#" + $grid.attr("id") + " > tbody > tr");
        len = rows.length;
        for (i = 0; i < len; i += 1) {
            rowData = $(rows[i]).data("row");
            if (rowData.id == id) {
                $row = $(rows[i]);
                break;
            }
        }
        return $row;
    },

    SelectRow: function ($grid, data, $row) {
        $row.addClass(data.style.content.rowSelected);
        /**
         * Event fires on selection of row
         *
         * @event rowSelect
         * @memberof Grid
         * @property {object} e - event data
         * @property {object} $row - the row presented as jquery object 
         * @property {string} id - the id of the record
         * @property {object} record - the data of the row record
         * @example <table id="grid"></table>
         * <script>
         *     var grid = $("#grid").grid({
         *         loader: { url: "/version_0_3/Demos/GetPlayers" },
         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
         *         selectionMethod: "checkbox"
         *     });
         *     grid.on("rowSelect", function (e, $row, id, record) {
         *         alert('Row with id=' + id + ' is selected.');
         *     });
         * </script>
         */
        $grid.trigger("rowSelect", [$row, $row.data("row").id, $row.data("row").record]);

        if ("checkbox" === data.selectionMethod) {
            $row.find("td:nth-child(1) input[type='checkbox']").prop("checked", true);
        }
    },

    UnselectRow: function ($grid, data, $row) {
        if ($row.hasClass(data.style.content.rowSelected)) {

            $row.removeClass(data.style.content.rowSelected);

            /**
             * Event fires on un selection of row
             *
             * @event rowUnselect
             * @memberof Grid
             * @property {object} e - event data
             * @property {object} $row - the row presented as jquery object 
             * @property {string} id - the id of the record
             * @property {object} record - the data of the row record
             * @example <table id="grid"></table>
             * <script>
             *     var grid = $("#grid").grid({
             *         loader: { url: "/version_0_3/Demos/GetPlayers" },
             *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
             *         selectionMethod: "checkbox"
             *     });
             *     grid.on("rowUnselect", function (e, $row, id, record) {
             *         alert('Row with id=' + id + ' is unselected.');
             *     });
             * </script>
             */
            $grid.trigger("rowUnselect", [$row, $row.data("row").id, $row.data("row").record]);

            if ("checkbox" === data.selectionMethod) {
                $row.find("td:nth-child(1) input[type='checkbox']").prop("checked", false);
            }
        }
    },

    _SetSelected: function ($grid, $row, id) {
        var data = $grid.data('grid');
        if ($row.hasClass(data.style.content.rowSelected)) {
            Grid.UnselectRow($grid, data, $row);
        } else {
            Grid.SelectRow($grid, data, $row);
        }
        if ("single" === data.selectionType) {
            $row.siblings().each(function () {
                Grid.UnselectRow($grid, data, $(this));
            });
        }
    },

    _GetSelected: function ($grid) {
        var result, data, selections;
        data = $grid.data("grid");
        selections = $grid.find("tbody > tr." + data.style.content.rowSelected);
        if (selections.length > 0) {
            result = $(selections[0]).data("row").id;
        }
        return result;
    },

    GetSelectedRows: function ($grid) {
        var data = $grid.data("grid");
        return $grid.find("tbody > tr." + data.style.content.rowSelected);
    },

    _GetSelections: function ($grid) {
        var result = [],
            $selections = Grid.GetSelectedRows($grid);
        if (0 < $selections.length) {
            $selections.each(function () {
                result.push($(this).data("row").id);
            });
        }
        return result;
    },

    GetRecordById: function ($grid, id) {
        var result = {}, rows, i, len, rowData;
        rows = $grid.find("tbody > tr");
        len = rows.length;
        for (i = 0; i < len; i += 1) {
            rowData = $(rows[i]).data("row");
            if (rowData.id === id) {
                result = rowData.record;
                break;
            }
        }
        return result;
    },

    GetRecordByIndex: function ($grid, index) {
        var result = {}, rows;
        rows = $grid.find("tbody > tr");
        if (rows.length > index) {
            result = $(rows[index]).data("row").record;
        }
        return result;
    },

    GetColumnPosition: function ($grid, index) {
        var i, result = -1, data = $grid.data("grid");
        for (i = 0; i < data.columns.length; i += 1) {
            if (data.columns[i].index === index) {
                result = i;
                break;
            }
        }
        return result;
    },

    GetColumnInfo: function ($grid, index) {
        var i, result = {}, data = $grid.data("grid");
        for (i = 0; i < data.columns.length; i += 1) {
            if (data.columns[i].index === index) {
                result = data.columns[i];
                break;
            }
        }
        return result;
    },

    GetCell: function ($grid, id, index) {
        var result = {}, rows, i, rowData, position;
        position = Grid.GetColumnPosition($grid, index);
        rows = $grid.find("tbody > tr");
        for (i = 0; i < rows.length; i += 1) {
            rowData = $(rows[i]).data("row");
            if (rowData.id === id) {
                result = $(rows[i].cells[position]).find("div");
                break;
            }
        }
        return result;
    },

    SetCellContent: function ($grid, id, index, value) {
        var column, $cellWrapper = Grid.GetCell($grid, id, index);
        $cellWrapper.empty();
        if (typeof (value) === "object") {
            $cellWrapper.append(value);
        } else {
            column = Grid.GetColumnInfo($grid, index);
            Grid.SetCellText($cellWrapper, column, value);
        }
    },

    CreatePager: function ($grid) {
        var $row, $cell, data, controls, $leftPanel, $rightPanel, $tfoot, $lblPageInfo;

        data = $grid.data('grid');

        if (data.pager.enable) {
            $row = $("<tr/>");
            $cell = $("<th/>").addClass(data.style.pager.cell);
            $cell.attr("colspan", $grid.find("thead > tr > th").length);
            $row.append($cell);

            $leftPanel = $("<div />").css({ "float": "left" });
            $rightPanel = $("<div />").css({ "float": "right" });
            if (/msie/.test(navigator.userAgent.toLowerCase())) {
                $rightPanel.css({ "padding-top": "3px" });
            }

            $cell.append($leftPanel).append($rightPanel);

            $tfoot = $("<tfoot />").append($row);
            $grid.append($tfoot);

            $.each(data.pager.leftControls, function () {
                $leftPanel.append(this);
            });

            $.each(data.pager.rightControls, function () {
                $rightPanel.append(this);
            });

            controls = $grid.find("TFOOT [data-role]");
            for (i = 0; i < controls.length; i++) {
                Grid.InitPagerControl($(controls[i]), $grid);
            }
        }
    },

    InitPagerControl: function ($control, $grid) {
        var data = $grid.data('grid')
        switch ($control.data("role")) {
            case "page-number":
                $control.bind("keypress", function (e) {
                    if (e.keyCode === 13) {
                        $(this).trigger("change");
                    }
                });
                break;
            case "page-size":
                if (data.pager.sizes && 0 < data.pager.sizes.length) {
                    $control.show();
                    $.each(data.pager.sizes, function () {
                        $control.append($("<option/>").attr("value", this.toString()).text(this.toString()));
                    });
                    $control.change(function () {
                        var newSize = parseInt(this.value, 10);
                        data.params[data.defaultParams.page] = 1;
                        Grid.SetPageNumber($grid, 1);
                        data.params[data.defaultParams.limit] = newSize;
                        $grid.reload();

                        /**
                         * Event fires on change of the page size
                         *
                         * @event pageSizeChange
                         * @memberof Grid
                         * @property {object} e - event data
                         * @property {int} newSize - The new page size
                         * @example <table id="grid"></table>
                         * <script>
                         *     var grid = $("#grid").grid({
                         *         loader: { url: "/version_0_3/Demos/GetPlayers" },
                         *         columns: [ { index: "ID" }, { index: "Name" }, { index: "PlaceOfBirth" } ],
                         *         pager: { enable: true, limit: 2, sizes: [2, 5, 10, 20] }
                         *     });
                         *     grid.on("pageSizeChange", function (e, newSize) {
                         *         alert('The new page size is ' + newSize + '.');
                         *     });
                         * </script>
                         */
                        $grid.trigger("pageSizeChange", [newSize]);
                    });
                    $control.val(data.params[data.defaultParams.limit]);
                } else {
                    $control.hide();
                }
                break;
            case "page-refresh":
                $control.bind("click", function () { $grid.reload(); });
                break;
        }

    },

    ReloadPager: function ($grid, data, response) {
        var totalRecords, page, limit, lastPage, firstRecord, lastRecord;

        totalRecords = response[data.mapping.totalRecordsField];
        if (!totalRecords || isNaN(totalRecords)) {
            totalRecords = 0;
        }
        if (data.pager.enable) {
            page = (0 === totalRecords) ? 0 : data.params[data.defaultParams.page];
            limit = parseInt(data.params[data.defaultParams.limit], 10);
            lastPage = Math.ceil(totalRecords / limit);
            firstRecord = (0 === page) ? 0 : (limit * (page - 1)) + 1;
            lastRecord = (firstRecord + limit) > totalRecords ? totalRecords : (firstRecord + limit) - 1;

            controls = $grid.find("TFOOT [data-role]");
            for (i = 0; i < controls.length; i++) {
                Grid.ReloadPagerControl($(controls[i]), $grid, page, lastPage, firstRecord, lastRecord, totalRecords);
            }

            $grid.find("TFOOT > TR > TH").attr("colspan", $grid.find("thead > tr > th").length);
        }
    },

    ReloadPagerControl: function ($control, $grid, page, lastPage, firstRecord, lastRecord, totalRecords) {
        var data = $grid.data('grid')
        switch ($control.data("role")) {
            case "page-first":
                if (page < 2) {
                    $control.addClass(data.style.pager.stateDisabled).unbind("click");
                } else {
                    $control.removeClass(data.style.pager.stateDisabled).unbind("click").bind("click", Grid.CreateFirstPageHandler($grid));
                }
                break;
            case "page-previous":
                if (page < 2) {
                    $control.addClass(data.style.pager.stateDisabled).unbind("click");
                } else {
                    $control.removeClass(data.style.pager.stateDisabled).unbind("click").bind("click", Grid.CreatePrevPageHandler($grid));
                }
                break;
            case "page-number":
                $control.val(page).unbind("change").bind("change", Grid.CreateChangePageHandler($grid, page, lastPage));
                break;
            case "page-label-last":
                $control.text(lastPage);
                break;
            case "page-next":
                if (lastPage === page) {
                    $control.addClass(data.style.pager.stateDisabled).unbind("click");
                } else {
                    $control.removeClass(data.style.pager.stateDisabled).unbind("click").bind("click", Grid.CreateNextPageHandler($grid));
                }
                break;
            case "page-last":
                if (lastPage === page) {
                    $control.addClass(data.style.pager.stateDisabled).unbind("click");
                } else {
                    $control.removeClass(data.style.pager.stateDisabled).unbind("click").bind("click", Grid.CreateLastPageHandler($grid, lastPage));
                }
                break;
            case "record-first":
                $control.text(firstRecord);
                break;
            case "record-last":
                $control.text(lastRecord);
                break;
            case "record-total":
                $control.text(totalRecords);
                break;
        }
    },

    CreateFirstPageHandler: function ($grid) {
        return function () {
            var data = $grid.data('grid');
            data.params[data.defaultParams.page] = 1;
            Grid.SetPageNumber($grid, 1);
            $grid.reload();
        };
    },

    CreatePrevPageHandler: function ($grid) {
        return function () {
            var data, currentPage, lastPage, newPage;
            data = $grid.data('grid');
            currentPage = data.params[data.defaultParams.page];
            newPage = (currentPage && currentPage > 1) ? currentPage - 1 : 1;
            data.params[data.defaultParams.page] = newPage;
            Grid.SetPageNumber($grid, newPage);
            $grid.reload();
        };
    },

    CreateNextPageHandler: function ($grid) {
        return function () {
            var data, currentPage;
            data = $grid.data('grid');
            currentPage = data.params[data.defaultParams.page];
            data.params[data.defaultParams.page] = currentPage + 1;
            Grid.SetPageNumber($grid, currentPage + 1);
            $grid.reload();
        };
    },

    CreateLastPageHandler: function ($grid, lastPage) {
        return function () {
            var data = $grid.data('grid');
            data.params[data.defaultParams.page] = lastPage;
            Grid.SetPageNumber($grid, lastPage);
            $grid.reload();
        };
    },

    CreateChangePageHandler: function ($grid, currentPage, lastPage) {
        return function (e) {
            var data = $grid.data('grid'),
                newPage = parseInt(this.value, 10);
            if (newPage && !isNaN(newPage) && newPage <= lastPage) {
                data.params[data.defaultParams.page] = newPage;
                $grid.reload();
            } else {
                this.value = currentPage;
                alert("Enter a valid page.");
            }
        };
    },

    SetPageNumber: function ($grid, value) {
        $grid.find("TFOOT [data-role='page-number']").val(value);
    },

    GetAll: function ($grid) {
        var result = [],
            rows = $grid.find("tbody > tr"),
            i = 0;

        for (; i < rows.length; i++) {
            result.push($(rows[i]).data("row"));
        }
        return result;
    }
};

(function ($) {
    $.fn.grid = function (method) {
        if (typeof method === 'object' || !method) {
            $.extend(this, Grid);
            return Grid.init.apply(this, arguments);
        } else if (Grid[method]) {
            return Grid[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else {
            $.error('Method ' + method + ' does not exist on jQuery.grid');
        }
    };
})(jQuery);
